from docplex.cp.model import *

from typing import List

from RulePenalty import *
from customTypes import *
from log import logger
from auxiliary import AuxiliaryStruct
from enum import Enum
from utils import rOr
from Parameter import Parameter
from patterns import *



class TipoRuleStudente(Enum):
    _1BucoOrario = 0
    _2BuchiOrario = 1
    SlotConsecutivi = 2

class RuleStudente:
    def __init__(self, eq_if, penalita:LV_Penalties, index_X_penaltiesStud:int, orientId:int, anno:int, day:int, tipoRuleStud:TipoRuleStudente,
                index_X_orient:int):
        ''''''
        self.eq_if = eq_if
        self.penalita:LV_Penalties = penalita
        self.index_X_penaltiesStud:int = index_X_penaltiesStud
        self.orientId:int = orientId
        self.periodoDidattico:str = str(anno) + "-1"
        self.day:int = day
        self.tipoRuleStud:TipoRuleStudente = tipoRuleStud
        # da dove inizia la lista delle 7 var binarie che indica gli slot temporali allocati nella giornata
        self.index_X_orient:int = index_X_orient 
        
        self.PARAM = Parameter()
        self.log:logger = logger()


class RuleStudenteHandler(metaclass=SingletonMeta):
    '''NOTE: questa classe usa delle equazioni particolari NON PARAMETRICHE -> non si possono settare dei limiti diversi a piacimento.
    Ogni modifica sui vincoli si traduce in una modifica più o meno complessa delle equazioni.\n
    FEATURES: 
        - 5/6 Day di lezione a settimana (in realtà questo lo posso variare senza dover riscrivere le equazioni)
        - 7 slot temporali di lezione al Day
        - max 6 Slot di lezione al Day -> Hard Constraint
        - max 5 Slot di lezione consecutivi 
        - 4 Slot di lezione consecutivi -> Soft Constraint LV_Penalties.LV_1
        - 5 Slot di lezione consecutivi -> Soft Constraint LV_Penalties.LV_2
        - 1 buco di 2 slot -> Soft Constraint LV_Penalties.LV_1
        - 1 buco di più di 2 slot -> Soft Constraint LV_Penalties.LV_4
        - 2 buchi nell'orario -> Soft Constraint LV_Penalties.LV_6
    '''
    
    def __init__(self, model:CpoModel):
        '''Si occupa di gestire i vincoli dal lato Studente:
        - model_AddRules_maxSlotOrientamentoPerDay():
            - il numero massimo di Slot allocabili in un Day per un dato Orientamento -> Hard
            - il numero massimo di Slot consecutivi in un Day per un dato Orientamento -> Hard
        - model_AddRules_X_orient():
            - si occupa di segnare in X_orient se i vari slot orari risultano allocati a qualche Insegnamento o meno
        - model_AddRules_X_penaltiesStud(): (e self.finalizeRules())
            - assegnare penalità in caso di buchi nell'Orario -> Soft
            - assegnare penalità in caso di troppi Slot consecutivi -> Soft
        '''
        self.AUX:AuxiliaryStruct = AuxiliaryStruct()
        self.PARAM = Parameter()
        self.log:logger = logger()
        self.load_X_orient:bool = False
        self.listRuleStudente:List[RuleStudente] = list()
                
        self.X_orient = model.binary_var_list(self.AUX.get_nPeriodiDidattici_Orientamenti()*self.AUX.get_nSlotSettimana(), "X_orient")
        '''Per tenere traccia dei piani di allocazione degli Slot appartenenti ai diversi Orientamenti, avrò 35 var binarie
        (5 Day di lezione * 7 slot temporali giornalieri) per ogni Anno di ogni Orientamento.\n
        
        1) gli Insegnamenti a cui si rivolge X_orient per visualizzarne l'allocazione nell'Orario sono soltanto quelli di tipo: 
        Obbligatorio, ObbligatorioAScelta e Suggerito (non disponibile con i dati attualmente in mio possesso).
        
        2) L'obiettivo è infatti quello di garantire la miglior soluzione lato Studente riguardo gli Insegnamenti caratterizzanti di un
        Orientamento; cercare di ottimizzare l'allocazione per Insegnamenti quali i CreditiLiberi, scelti da Studenti appartenenti ad 
        Orientamenti diversi, rischierebbe di mettere in ombra il problema principale.
        
        3) X_orient[n] = 1 indica che quello slot temporale è allocato per ALMENO un Insegnamento di un tipo tra quelli indicati al punto 1), 
        X_orient[n] = 0 indica che non è allocato NESSUN Insegnamento di un tipo tra quelli indicati al punto 1) nello slot temporale corrente.
        I valori di n variano tenendo conto dell'Orientamento, dell'Anno, del Day della settimana e dello slot orario.\n
        
        4) APPROSSIMAZIONE: per raggiungere un tradeoff ottimale tra rappresentazione corretta dei dati e dimensione aggiuntiva del modello
        non si tiene conto di Slot appartenenti allo stesso Orientamento ed allocabili in parallelo tra loro.
        (eg: diverse squadre di laboratorio tenute in Locali diversi da Docenti diversi, differenti Alfabetiche dello stesso Insegnamento ed
        Insegnamento erogato in italiano ed in inglese). Ciò NON SIGNIFICA che la soluzione ottenuta nel caso si andasse a minimizzare i costi
        dati da queste penalità non sarebbe un risultato ottimo, bensì significa che le soluzione esplorate saranno un sottoinsieme
        di quelle realmente esplorabili (la differenza dei due insiemi è piccola dato che spesso gli Insegnamenti allocabili in parallelo
        verranno veramente allocati in parallelo da CPLEX).\n
        La differenziazione tra il modello esatto e quello approssimato si ha solo quando Insegnamenti parallelizzabili non saranno allocati
        in parallelo e quindi avrò due entry di X_orient[] settate a 1 quando in realtà nella stessa giornata se quegli Slot fossero allocati
        in parallelo avrei uno slot temporale in meno di lezione.\n
        L'eliminazione completa di questa approssimazione richiederebbe un aumento delle variabili definite per questa parte di modello di
        circa 4 volte e un appesantimento delle equazioni per il modello non indifferente, resta comunque un'opzione realizzabile anche
        se a mio avviso i benefici sarebbero decisamente superati dai maggiori tempi di esecuzione\n
        '''        
        

    def helper_getOffsetOrient(self, orientId:int, anno:int) -> int:
        '''Dato l'Anno in cui si svolge l'Insegnamento (in realtà la granularità è a livello di Slot) e l'Orientamento, ritorna l'offset nella
        lista self.X_orient.\n
        NB: qui per ogni Anno (per ogni Orientamento) si ha una variabile per ogni slot temporale (ie 35 variabili per ogni Anno, per ogni
        Orientamento).\n
        Params:
            Anno (int) in [1,2,3]
            orientId (int) tra quelli previsti in self.AUX'''
        nRes:int = 0
        for i in range(orientId):
            if self.AUX.list_Orientamenti[i].tipoCdl == TipoCdl.Triennale:
                nRes += 3*self.AUX.get_nSlotSettimana()
            elif self.AUX.list_Orientamenti[i].tipoCdl == TipoCdl.Magistrale:
                nRes += 2*self.AUX.get_nSlotSettimana()
            else:
                self.log.error_log("constraintBuilder.helper_getOffsetOrient(): TipoCdl sconosciuto")
        return nRes + (anno-1)*self.AUX.get_nSlotSettimana() # ritorno la posizione tenendo conto anche dell'anno per lo Slot corrente
    
    def helper_getOffsetPenalita(self, orientId:int, anno:int, day:int) -> int:
        '''Dato l'Anno in cui si svolge l'Insegnamento (in realtà la granularità è a livello di Slot), l'Orientamento e il giorno, 
        ritorna l'offset nella lista self.X_penalties.\n
        NB: qui per ogni Anno si ha UNA SOLA variabile per ogni Day (ie 5 variabili per ogni Anno, per il singolo Orientamento)\n
        Params:
            Anno (int) in [1,2,3]
            orientId (int) tra quelli previsti in self.AUX
            day (int) in [0, .., 4]'''
        nRes:int = 0
        for i in range(orientId):
            if self.AUX.list_Orientamenti[i].tipoCdl == TipoCdl.Triennale:
                nRes += 3*self.AUX.get_NUM_DAY()
            elif self.AUX.list_Orientamenti[i].tipoCdl == TipoCdl.Magistrale:
                nRes += 2*self.AUX.get_NUM_DAY()
            else:
                self.log.error_log("constraintBuilder.helper_getOffsetPenalita(): TipoCdl sconosciuto")
        return nRes + (anno-1)*self.AUX.get_NUM_DAY() + day # Tengo conto anche dell'anno per lo slot corrente    
    
    
    def model_AddRules_X_orient(self, X_d, X_h):
        '''Crea le equazioni che si occupano di impostare i valori per le variabili del modello X_orient'''
        eqs = list()
        
        # implies su X_orient
        for orientId in self.AUX.map_Orientamento_to_IdOrientamento.values():
            # per ogni Orientamento
            for index_slot in range(len(self.AUX.list_slotIdInOrientamento[orientId])):
                # se lo Slot per l'Orientamento è Obbligatorio/ObbligatorioAScelta
                if self.AUX.list_slotIdInOrientamento_tipo[orientId][index_slot] not in [TipoInsegnamento.Obbligatorio, TipoInsegnamento.ObbligatorioAScelta]:
                    continue
                
                # per ogni Slot nell'Orientamento
                slotId = self.AUX.list_slotIdInOrientamento[orientId][index_slot]

                for day in range(self.AUX.get_NUM_DAY()):
                    for hour in range(self.AUX.NUM_SLOT_PER_DAY-self.AUX.pianoAllocazione[slotId].numSlot+1): # 0..6
                        for nSlot_durata in range(self.AUX.pianoAllocazione[slotId].numSlot): # 0..numSlot-1
                            
                            # l'eq è specifica del PeriodoDidattico dello Slot
                            if self.AUX.list_slotIdInOrientamento_periodoDidattico[orientId][index_slot] in [PeriodoDidattico.PrimoAnno_PrimoSemestre, PeriodoDidattico.PrimoAnno_SecondoSemestre]:
                                eq = if_then(logical_and(X_d[slotId] == day, X_h[slotId] == hour),
                                            self.X_orient[self.helper_getOffsetOrient(orientId,1)+day*self.AUX.NUM_SLOT_PER_DAY+hour+nSlot_durata] == 1)

                            elif self.AUX.list_slotIdInOrientamento_periodoDidattico[orientId][index_slot] in [PeriodoDidattico.SecondoAnno_PrimoSemestre, PeriodoDidattico.SecondoAnno_SecondoSemestre]:
                                eq = if_then(logical_and(X_d[slotId] == day, X_h[slotId] == hour),
                                            self.X_orient[self.helper_getOffsetOrient(orientId,2)+day*self.AUX.NUM_SLOT_PER_DAY+hour+nSlot_durata] == 1)
                            
                            elif self.AUX.list_slotIdInOrientamento_periodoDidattico[orientId][index_slot] in [PeriodoDidattico.TerzoAnno_PrimoSemestre, PeriodoDidattico.TerzoAnno_SecondoSemestre]:
                                eq = if_then(logical_and(X_d[slotId] == day, X_h[slotId] == hour),
                                            self.X_orient[self.helper_getOffsetOrient(orientId,3)+day*self.AUX.NUM_SLOT_PER_DAY+hour+nSlot_durata] == 1)
                                                       
                            eqs.append(eq)   
        self.load_X_orient = True
        return eqs                    
            
    def model_AddRules_maxSlotOrientamentoPerDay(self, model:CpoModel):
        '''Imposta il numero di Slot relativi a Insegnamenti di tipo Obbligatorio, ObbligatorioAScelta, Suggerito(non disponibile) che
        possono essere allocati in un singolo Day.\n
        EURISTICA: si stabilisce a livello di codice che il numero di Slot giornalieri è 6 e che il numero massimo di Slot consecutivi 
        giornalieri è 5. Se si volessero usare limiti diversi bisognerebbe riscrivere le equazioni.\n
        Questo perchè sapendo che in una giornata ci sono massimo 7 slot allocabili, imporre il numero massimo di slot consecutivi debba
        essere 5, sapendo che il numero massimo complessivo di slot giornalieri è 6, si traduce nel dire semplicemente che gli slot
        allocati in un giorno sono 6 => il primo e l'ultimo di giornata devono essere allocati, che significa che il "buco" sarà all'interno
        della giornata
       '''
        eqs = list()
        if not self.load_X_orient:
            self.log.error_log("RuleStudente.model_AddRules_maxSlotOrientamentoPerDay(): impossibile creare le equazioni, mancano equazioni su X_orient")
            return eqs
        
        # sums su X_orient
        for orientId in self.AUX.map_Orientamento_to_IdOrientamento.values():
            # per ogni Orientamento
            
            # fix troppi SlotScelti
            if self.AUX.list_Orientamenti[orientId].nomeCdl == "INGEGNERIA ELETTRONICA" and self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Triennale:
                continue
            if self.AUX.list_Orientamenti[orientId].nomeCdl == "INGEGNERIA FISICA" and self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Triennale:
                continue        
            # if self.AUX.list_Orientamenti[orientId].nomeCdl == "COMMUNICATIONS AND COMPUTER NETWORKS ENGINEERING (INGEGNERIA TELEMATICA E DELLE COMUNICAZIONI)" and self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Triennale:
                # continue
                            
            for day in range(self.AUX.get_NUM_DAY()):
                # Per ogni orientamento non ci devono essere più di 6 slot al giorno
                
                if self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Triennale:
                    # TerzoAnno dell'Orientamento
                    eqMax = model.sum(self.X_orient[self.helper_getOffsetOrient(orientId,3)+day*self.AUX.NUM_SLOT_PER_DAY+i] == 1 for i in range(self.AUX.NUM_SLOT_PER_DAY)) <= 6
                    # se ho 6 slot allocati => il primo e l'ultimo della giornata devono essere entrambi allocati
                    # ie per avere l'ora buca non agli estremi della giornata per avere max 5 slot consecutivi 
                    # -> se cambia il numero di Slot giornalieri devo riscrivere questi vincoli
                    eqCons = if_then(model.sum(self.X_orient[self.helper_getOffsetOrient(orientId,3)+day*self.AUX.NUM_SLOT_PER_DAY+i] == 1 for i in range(self.AUX.NUM_SLOT_PER_DAY)) == 6,
                                     logical_and(self.X_orient[self.helper_getOffsetOrient(orientId,3)+day*self.AUX.NUM_SLOT_PER_DAY] == 1,
                                                 self.X_orient[self.helper_getOffsetOrient(orientId,3)+day*self.AUX.NUM_SLOT_PER_DAY+self.AUX.NUM_SLOT_PER_DAY-1] == 1))
                    eqs.extend([eqMax,eqCons])
                
                # PrimoAnno dell'Orientamento
                eq1Max = model.sum(self.X_orient[self.helper_getOffsetOrient(orientId,1)+day*self.AUX.NUM_SLOT_PER_DAY+i] == 1 for i in range(self.AUX.NUM_SLOT_PER_DAY)) <= 6
                eq1Cons = if_then(model.sum(self.X_orient[self.helper_getOffsetOrient(orientId,1)+day*self.AUX.NUM_SLOT_PER_DAY+i] == 1 for i in range(self.AUX.NUM_SLOT_PER_DAY)) == 6,
                    logical_and(self.X_orient[self.helper_getOffsetOrient(orientId,1)+day*self.AUX.NUM_SLOT_PER_DAY] == 1,
                                self.X_orient[self.helper_getOffsetOrient(orientId,1)+day*self.AUX.NUM_SLOT_PER_DAY+self.AUX.NUM_SLOT_PER_DAY-1] == 1))

                # SecondoAnno dell'Orientamento
                eq2Max = model.sum(self.X_orient[self.helper_getOffsetOrient(orientId,2)+day*self.AUX.NUM_SLOT_PER_DAY+i] == 1 for i in range(self.AUX.NUM_SLOT_PER_DAY)) <= 6
                eq2Cons = if_then(model.sum(self.X_orient[self.helper_getOffsetOrient(orientId,2)+day*self.AUX.NUM_SLOT_PER_DAY+i] == 1 for i in range(self.AUX.NUM_SLOT_PER_DAY)) == 6,
                    logical_and(self.X_orient[self.helper_getOffsetOrient(orientId,2)+day*self.AUX.NUM_SLOT_PER_DAY] == 1,
                                self.X_orient[self.helper_getOffsetOrient(orientId,2)+day*self.AUX.NUM_SLOT_PER_DAY+self.AUX.NUM_SLOT_PER_DAY-1] == 1))
                
                eqs.extend([eq1Max,eq1Cons,eq2Max,eq2Cons])                
        return eqs
    


    
    def finalizeRules(self, model:CpoModel) -> int:
        '''Definisce i SoftConstraint per:
            - numero di Slot consecutivi tra quelli Obbligatori, ObbligatoriAScelta di una dato Orientamento
            - numero limite di "buchi" nell'orario tenendo conto degli Slot di Insegnamenti Obbligatori,ObbligatoriAScelta.\n
            DIMENSIONI: ho 2 entry di penalità per ogni giorno di ogni Anno di un Orientamento, per ogni Orientamento
            -> una per i buchi nell'Orario e una per gli Slot consecutivi.\n
            Return:
                Il numero di variabili per tenere conto delle penalità.'''
        nVarSlotConsecutivi:int = 0
        nVar1Buco:int = 0
        nVar2Buchi:int = 0
        
        if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Studenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Studenti == 0:
            if self.PARAM.debuggingPenalty == False:
                return 0
        
        # limite Slot consecutivi
        for orientId in self.AUX.map_Orientamento_to_IdOrientamento.values():
            # Per ogni Orientamento se ci sono 3 Slot consecutivi => penalità
                
            for year in range(1,4):
                if year == 3 and self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Magistrale:
                    break

                for day in range(self.AUX.get_NUM_DAY()):
                    
                    # 4 Slot consecutivi -> penalità: LV_1
                    listOptEq = list()
                    for hStart in range(self.AUX.NUM_SLOT_PER_DAY-3):
                        eq_if = model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + hStart+i] == 1
                                        for i in range(4)) == 4
                        listOptEq.append(eq_if)

                    self.listRuleStudente.append(RuleStudente(rOr(listOptEq), LV_Penalties.LV_1, 
                                                self.helper_getOffsetPenalita(orientId, year, day),
                                                orientId, year, day, TipoRuleStudente.SlotConsecutivi,
                                                self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY))
                
                    # 5 Slot consecutivi -> penalità: LV_2
                    listOptEq = list()                    
                    for hStart in range(self.AUX.NUM_SLOT_PER_DAY-4):
                        eq_if = model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + hStart+i] == 1
                                        for i in range(5)) == 5
                        listOptEq.append(eq_if)

                    self.listRuleStudente.append(RuleStudente(rOr(listOptEq), LV_Penalties.LV_2, 
                                                self.helper_getOffsetPenalita(orientId, year, day),
                                                orientId, year, day, TipoRuleStudente.SlotConsecutivi,
                                                self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY))                            
                    
                    nVarSlotConsecutivi += 1
                    
        # limite 1 buco nell'orario
        for orientId in self.AUX.map_Orientamento_to_IdOrientamento.values():
            
            for year in range(1,4):
                if year == 3 and self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Magistrale:
                    break
                                
                for day in range(self.AUX.get_NUM_DAY()):
                    
                    # 1 buco di 2 slot nell'orario
                    listOptEq = list()                    
                    for buco in range(1,self.AUX.NUM_SLOT_PER_DAY-2):
                        eq_if = logical_and(model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                        for i in range(buco)) 
                                == model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                        for i in range(buco+2))
                            , 
                                model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                        for i in range(buco)) 
                                != model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + j] 
                                        for j in range(buco+3))) 
                                        # no self.AUX.NUM_SLOT_PER_DAY perchè altrimenti saprei solo se ho un buco di ALMENO 3 slot
                        listOptEq.append(eq_if)

                    self.listRuleStudente.append(RuleStudente(rOr(listOptEq), LV_Penalties.LV_1, 
                                                nVarSlotConsecutivi + self.helper_getOffsetPenalita(orientId, year, day),
                                                orientId, year, day, TipoRuleStudente._1BucoOrario,
                                                self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY))                             
                    
                    # 1 buco di più di 2 slot nell'orario
                    listOptEq = list()                    
                    for buco in range(1,self.AUX.NUM_SLOT_PER_DAY-3):
                        eq_if = logical_and(
                                model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                        for i in range(buco)) 
                                == model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                        for i in range(buco+3)) # implica un buco di almeno 3 slot (anche di più eventualmente)
                            , 
                                model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                        for i in range(buco)) 
                                != model.sum(
                                    self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + j] 
                                        for j in range(self.AUX.NUM_SLOT_PER_DAY)))
                        listOptEq.append(eq_if)

                    self.listRuleStudente.append(RuleStudente(rOr(listOptEq), LV_Penalties.LV_4, 
                                                nVarSlotConsecutivi + self.helper_getOffsetPenalita(orientId, year, day),
                                                orientId, year, day, TipoRuleStudente._1BucoOrario,
                                                self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY))                                  
                    nVar1Buco += 1
                                        
        # 2 buchi nell'orario
        for orientId in self.AUX.map_Orientamento_to_IdOrientamento.values():
            
            for year in range(1,4):
                if year == 3 and self.AUX.list_Orientamenti[orientId].tipoCdl == TipoCdl.Magistrale:
                    break
                                
                for day in range(self.AUX.get_NUM_DAY()):                    
                    
                    for buco1 in range(1,self.AUX.NUM_SLOT_PER_DAY-1-2):
                        for buco2 in range(buco1+2, self.AUX.NUM_SLOT_PER_DAY-1):
                            eq_if = logical_and(
                                    logical_and(
                                        logical_and(
                                            model.sum(
                                                self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                    for i in range(buco1))
                                            == model.sum(
                                                self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                    for i in range(buco1+1))
                                        ,
                                            model.sum(
                                                self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                    for i in range(buco1))
                                            != model.sum(
                                                self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                    for i in range(buco2))
                                        )
                                    ,
                                        model.sum(
                                            self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                for i in range(buco2))
                                        == model.sum(
                                            self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                for i in range(buco2+1))
                                    )
                                ,
                                        model.sum(
                                            self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                            for i in range(buco2))
                                        !=  model.sum(
                                            self.X_orient[self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY + i] 
                                                for i in range(self.AUX.NUM_SLOT_PER_DAY))
                                )

                            self.listRuleStudente.append(RuleStudente(eq_if, LV_Penalties.LV_6, 
                                                nVarSlotConsecutivi + nVar1Buco + self.helper_getOffsetPenalita(orientId, year, day),
                                                orientId, year, day, TipoRuleStudente._2BuchiOrario,
                                                self.helper_getOffsetOrient(orientId,year)+day*self.AUX.NUM_SLOT_PER_DAY))                                          
                    nVar2Buchi += 1
                            
        return nVar1Buco + nVarSlotConsecutivi + nVar2Buchi
            
            
    def model_AddRules_X_penaltiesStud(self, index:int, X_penaltiesStud):
        '''Aggiunge le RuleStudente rigurardanti i Soft Constraint elencati.\n
        In un dato Orientamento-Anno-Day posso avere:
            - 4 slot consecutivi OR 5 slot consecutivi
            - 1/2 buchi da 2 slot OR 1 buco da 3 o più slot (non esiste la possibilità di avere 1 buco da 2 slot e 1 buco da 3 slot insieme)
            - 2 buchi nell'orario
        '''
        
        eqs = list()    
        
        # se non mi interessano le penalità dei soft constraint (e quindi non ho imposto nessun vincolo su di essi)
        # non aggiungo questa equazioni al modello
        if self.PARAM.additionalModelConfig & Parameter.OptimizeComponent.Studenti == 0 and self.PARAM.modelConfig & Parameter.OptimizeComponent.Studenti == 0:
            if self.PARAM.debuggingPenalty == False:
                # self.log.info_log("skip penalty Studente")
                return eqs
        
        
        for index_i in range(len(self.listRuleStudente)):
            for index_j in range(len(self.listRuleStudente)):
                if index_j > index_i:
                    rule_i:RuleStudente = self.listRuleStudente[index_i]
                    rule_j:RuleStudente = self.listRuleStudente[index_j]
                    
                    if rule_i.tipoRuleStud == TipoRuleStudente.SlotConsecutivi and rule_j.tipoRuleStud == TipoRuleStudente.SlotConsecutivi:
                        # se si riferiscono allo stesso Orientamento-Anno-Day => si riferiscono alla stessa var penalità
                        if rule_i.index_X_penaltiesStud != rule_j.index_X_penaltiesStud:
                            continue
                        # le due Rule sono mutuamente esclusive: se ho 4 slot consecutivi non ne ho 5 e viceversa
                        eq = if_then(rule_i.eq_if, 
                            X_penaltiesStud[rule_i.index_X_penaltiesStud+index] >= getIntFromLV_Penalties(rule_i.penalita, TipoPenalty.Studente))
                        eq1 = if_then(rule_j.eq_if, 
                            X_penaltiesStud[rule_j.index_X_penaltiesStud+index] >= getIntFromLV_Penalties(rule_j.penalita, TipoPenalty.Studente))
                        eq2 = if_then(
                            logical_not(
                                logical_or(rule_i.eq_if,rule_j.eq_if)
                            ), X_penaltiesStud[rule_j.index_X_penaltiesStud+index] == 0
                        )
                        eqs.extend([eq, eq1])
                    
                    if rule_i.tipoRuleStud == TipoRuleStudente._1BucoOrario and rule_j.tipoRuleStud == TipoRuleStudente._1BucoOrario:
                        # se si riferiscono allo stesso Orientamento-Anno-Day => si riferiscono alla stessa var penalità
                        if rule_i.index_X_penaltiesStud != rule_j.index_X_penaltiesStud:
                            continue
                        # le due Rule sono mutuamente esclusive: se ho 1 buco da 2 Slot non ne ho uno da più di 2 Slot
                        eq = if_then(rule_i.eq_if, 
                            X_penaltiesStud[rule_i.index_X_penaltiesStud+index] >= getIntFromLV_Penalties(rule_i.penalita, TipoPenalty.Studente))
                        eq1 = if_then(rule_j.eq_if, 
                            X_penaltiesStud[rule_j.index_X_penaltiesStud+index] >= getIntFromLV_Penalties(rule_j.penalita, TipoPenalty.Studente))
                        eq2 = if_then(
                            logical_not(
                                logical_or(rule_i.eq_if,rule_j.eq_if)
                            ), X_penaltiesStud[rule_j.index_X_penaltiesStud+index] == 0
                        )                        
                        eqs.extend([eq, eq1])    
                    
        # Rule 2 buchi
        for rule in self.listRuleStudente:
            if rule.tipoRuleStud == TipoRuleStudente._2BuchiOrario:
                eq = if_then(rule.eq_if, 
                        X_penaltiesStud[rule.index_X_penaltiesStud+index] == getIntFromLV_Penalties(rule.penalita, TipoPenalty.Studente))
                eq1 = if_then(logical_not(rule.eq_if),
                        X_penaltiesStud[rule.index_X_penaltiesStud+index] == 0)
            eqs.extend([eq1, eq])
     
        return eqs